#!/usr/bin/bash
set -x

# Description:
# This script is for installing forgejo-runner. To use it, you need to execute it with a version as an argument
#
# For example:
#
#     forgejo-runner-install 5.0.3

# functions
# example: rm / || die 'It didn't work!' 2
die() {
    local message=$1
    local code=$2

    echo -e "Error: $1"
    exit $2
}


version=$1
url="https://code.forgejo.org/forgejo/runner/releases/download/v$version/forgejo-runner-$version-linux-amd64"
binary="forgejo-runner-$version-linux-amd64"

# check if version was provided
if [[ -z $version ]]; then
    die "You didn't provide a version to install. Please, provide one similar to: a.b.c" 1
fi

if ! command -v podman &> /dev/null; then
	die "Podman is not installed. Please, install it and reboot before proceeding." 2
fi

# download and symlink
cd /usr/local/bin
curl -LO "$url"
ln -fs $binary forgejo-runner
cd -

# fix perms
chmod 755 /usr/local/bin/$binary

# create user
if ! id -u forgejo-runner; then
    useradd -mr -d /var/lib/forgejo-runner forgejo-runner

	# enable podman
	systemctl --user -M forgejo-runner@ enable --now podman.socket

	# enable linger
	loginctl enable-linger forgejo-runner

	# create systemd service
	user_home=$( sudo -iu forgejo-runner pwd )
	user_service_path=$user_home/.config/systemd/user
	sudo -u forgejo-runner mkdir -p $user_service_path

	if [[ ! -f $user_service_path/forgejo-runner.service ]]; then
		cat <<- 'EOF' | sudo -u forgejo-runner tee $user_service_path/forgejo-runner.service
		[Unit]
		Description=Forgejo Runner
		Documentation=https://forgejo.org/docs/latest/admin/actions/

		[Service]
		ExecStart=forgejo-runner daemon
		ExecReload=/bin/kill -s HUP $MAINPID
		WorkingDirectory=/var/lib/forgejo-runner
		Restart=on-failure
		TimeoutSec=0
		RestartSec=10

		[Install]
		WantedBy=multi-user.target
		EOF

		# activate service
		systemctl --user -M forgejo-runner@ daemon-reload
		systemctl --user -M forgejo-runner@ enable --now forgejo-runner.service
	else
		# reload
		systemctl --user -M forgejo-runner@ reload forgejo-runner.service
	fi
fi

# fix perms
chmod -R o-rwx /var/lib/forgejo-runner

# registration
echo 'Now, you need to proceed registering forgejo-runner'.